/* batch_ScoreAnimations2-FG-64.jsx — 64x32 FG Animation #2
 * - Sets TeamName text to team abbreviation with Primary fill color
 * - Sets FIELD GOAL text to Primary fill color
 * - Replaces TeamLogo 2 with team logo
 * - Sets TeamColorSolid to Primary color
 */

(function () {
  // --- load common helpers ---
  (function(){
    var COMMON = $.getenv("AE_COMMON_JSX") || "";
    if (COMMON) {
      try { delete $.global.GL; } catch(e) { $.global.GL = undefined; }
      $.evalFile(File(COMMON));
    }
    if (!$.global.GL) throw new Error("Common library not loaded. Set AE_COMMON_JSX to gl_common.jsxinc");
  })();

  var GL = $.global.GL;

  // --- env & defaults ---
  var PROJECT   = GL.env("AE_PROJECT", null);
  var CSV_PATH  = GL.env("AE_CSV", null);
  var COMP_NAME = GL.env("AE_COMP", "Comp-FG-64x32");

  // Layer names
  var TEAMNAME_LAYER    = GL.cleanValue(GL.env("AE_TEAMNAME_LAYER", "TeamName"));
  var FIELDGOAL_LAYER   = GL.cleanValue(GL.env("AE_FIELDGOAL_LAYER", "FIELD GOAL"));
  var TEAMLOGO_LAYER    = GL.cleanValue(GL.env("AE_TEAMLOGO_LAYER", "TeamLogo 2"));
  var TEAMCOLOR_SOLID   = GL.cleanValue(GL.env("AE_TEAMCOLOR_SOLID", "TeamColorSolid"));

  var LEAGUE    = GL.env("AE_LEAGUE", "");
  var LIMIT_STR = GL.env("AE_LIMIT", "");
  var LIMIT     = (LIMIT_STR && !isNaN(parseInt(LIMIT_STR,10))) ? parseInt(LIMIT_STR,10) : null;

  var OUTDIR    = GL.env("AE_OUTDIR", "");
  var PATH_TPL  = GL.env("AE_PATH_TEMPLATE", "{league}");
  var ANIM_NAME = GL.env("AE_ANIM", "_FG_2_1");
  var RS_TPL    = GL.env("AE_RS_TEMPLATE", "Best Settings");
  var OM_TPL    = GL.env("AE_OM_TEMPLATE", "PNG Sequence");
  var PURGE     = (GL.env("AE_PURGE_BEFORE_RENDER", "1") === "1");
  var NO_RENDER = (GL.env("AE_NO_RENDER", "0") === "1");
  var QUIT_APP  = (GL.env("AE_QUIT", "1") === "1");

  var logoOpts = {
    dir: GL.env("AE_LOGO_DIR", ""),
    tpl: GL.env("AE_LOGO_PATH_TEMPLATE", "{league}/{abbr}"),
    exts: GL.env("AE_LOGO_EXTS", "png,jpg,jpeg,svg,ai,psd")
  };

  // --- start ---
  if (app.beginSuppressDialogs) { try { app.beginSuppressDialogs(); } catch(e) {} }
  app.beginUndoGroup("FG-64 #2 - Batch Render");

  if (!PROJECT) GL.fail("AE_PROJECT env not set.");
  var aep = new File(PROJECT);
  if (!aep.exists) GL.fail("AE_PROJECT not found: " + PROJECT);
  try {
    var already = app.project && app.project.file && (app.project.file.fsName === aep.fsName);
    if (!already) app.open(aep);
  } catch(e) { app.open(aep); }

  if (!CSV_PATH) GL.fail("AE_CSV env not set.");
  if (!LEAGUE || GL.cleanValue(LEAGUE) === "") GL.fail("AE_LEAGUE is required.");

  var rows  = GL.parseCSV(GL.openRead(CSV_PATH));
  var teams = GL.buildTeams(rows, LEAGUE);
  var todo  = GL.pickTeamsLeagueOnly(teams, LEAGUE);
  if (LIMIT && todo.length > LIMIT) todo = todo.slice(0, LIMIT);
  if (!todo.length) GL.fail("No teams matched league: " + LEAGUE);

  var comp = GL.findComp(COMP_NAME);
  if (!comp) GL.fail("Comp not found: " + COMP_NAME);

  var rootOut = OUTDIR ? new Folder(OUTDIR) : (app.project.file ? app.project.file.parent : Folder.desktop);
  GL.ensureFolder(rootOut);

  GL.rqClear();

  // --- helper functions ---

  // Set TeamName text and fill color to primary
  function setTeamNameTextAndFill(compItem, abbr, color01) {
    var layer = GL.getLayer(compItem, TEAMNAME_LAYER);
    if (!layer) {
      $.writeln("WARNING: TeamName layer not found: " + TEAMNAME_LAYER);
      return false;
    }
    GL.setTextContent(layer, abbr);
    return GL.setTextOrFillColor(layer, color01);
  }

  // Set FIELD GOAL text fill color to primary
  function setFieldGoalTextColor(compItem, color01) {
    var layer = GL.getLayer(compItem, FIELDGOAL_LAYER);
    if (!layer) {
      $.writeln("WARNING: FIELD GOAL layer not found: " + FIELDGOAL_LAYER);
      return false;
    }
    return GL.setTextOrFillColor(layer, color01);
  }

  // Set TeamColorSolid to primary color
  function setTeamColorSolid(compItem, color01) {
    var layer = GL.getLayer(compItem, TEAMCOLOR_SOLID);
    if (!layer) {
      $.writeln("WARNING: TeamColorSolid layer not found: " + TEAMCOLOR_SOLID);
      return false;
    }
    return GL.setSolidColor(layer, color01);
  }

  // Import and replace logo
  function findOrImportFootage(filePath) {
    var f = new File(filePath);
    if (!f.exists) return null;
    var fs = f.fsName;

    // Check if already imported
    for (var i = 1; i <= app.project.numItems; i++) {
      var it = app.project.item(i);
      try {
        if (it instanceof FootageItem && it.file && it.file.fsName === fs) return it;
      } catch(e) {}
    }

    // Import new
    try {
      var io = new ImportOptions(f);
      if (io.canImportAs(ImportAsType.FOOTAGE)) {
        io.importAs = ImportAsType.FOOTAGE;
        return app.project.importFile(io);
      }
    } catch(e) {}
    return null;
  }

  // --- per team loop ---
  $.writeln("\n========================================");
  $.writeln("Processing " + todo.length + " teams for league: " + LEAGUE);
  $.writeln("Comp: " + COMP_NAME);
  $.writeln("Animation: " + ANIM_NAME);
  $.writeln("========================================\n");

  for (var i = 0; i < todo.length; i++) {
    var t = todo[i];
    $.writeln("["+(i+1)+"/"+todo.length+"] Processing: " + t.abbr + " (" + t.name + ")");

    var smart = GL.computeSmartColors(GL.safeColor(t.primary), GL.safeColor(t.secondary));
    var P = smart.primary;

    // 1) Set TeamName text and fill to Primary
    if (setTeamNameTextAndFill(comp, t.abbr, P)) {
      $.writeln("  ✓ Set TeamName: " + t.abbr + " (Primary color)");
    }

    // 2) Set FIELD GOAL text to Primary
    if (setFieldGoalTextColor(comp, P)) {
      $.writeln("  ✓ Set FIELD GOAL to Primary color");
    }

    // 3) Set TeamColorSolid to Primary
    if (setTeamColorSolid(comp, P)) {
      $.writeln("  ✓ Set TeamColorSolid to Primary color");
    }

    // 4) Replace TeamLogo 2
    var logoFile = GL.findLogoFile(t.league, t.abbr, logoOpts);
    if (logoFile) {
      $.writeln("  → Found logo: " + logoFile.fsName);
      var footage = findOrImportFootage(logoFile.fsName);

      if (footage) {
        $.writeln("  ✓ Imported logo: " + footage.name);

        var logoLayer = GL.getLayer(comp, TEAMLOGO_LAYER);
        if (logoLayer) {
          // Check if it's a precomp
          if (logoLayer.source && (logoLayer.source instanceof CompItem)) {
            $.writeln("  → TeamLogo 2 is a precomp, replacing footage inside...");
            var precomp = logoLayer.source;
            var replaced = 0;

            for (var L = 1; L <= precomp.numLayers; L++) {
              var subLayer = precomp.layer(L);
              try {
                if (subLayer.source && (subLayer.source instanceof FootageItem)) {
                  subLayer.replaceSource(footage, false);
                  $.writeln("  ✓ Replaced '" + subLayer.name + "' inside TeamLogo 2 precomp");
                  replaced++;
                }
              } catch(e) {
                $.writeln("  ✗ Error replacing '" + subLayer.name + "': " + e);
              }
            }

            if (replaced === 0) {
              $.writeln("  ✗ No footage layers found inside TeamLogo 2 precomp");
            }
          } else if (logoLayer.source) {
            // Direct footage layer
            try {
              logoLayer.replaceSource(footage, false);
              $.writeln("  ✓ Replaced TeamLogo 2");
            } catch(e) {
              $.writeln("  ✗ Error replacing TeamLogo 2: " + e);
            }
          } else {
            $.writeln("  ✗ TeamLogo 2 has no source");
          }
        } else {
          $.writeln("  ✗ TeamLogo 2 layer not found");
        }
      }
    } else {
      $.writeln("  ✗ Logo file not found for " + t.abbr);
    }

    // Pixel snap
    GL.snapCompTextForPixelArt(comp);

    if (PURGE && app.purge) {
      try { app.purge(PurgeTarget.ALL_CACHES); } catch(e) {}
    }

    // Render
    if (!NO_RENDER) {
      var lc = GL.leagueAndConfForPath(t.league, t.conference);
      var paths = GL.outPaths(rootOut, PATH_TPL, lc.base, t.abbr, ANIM_NAME, lc.conf, t.espn_team_id);
      $.writeln("  → Output: " + paths.file.fsName);
      GL.rqRenderTo(comp, RS_TPL, OM_TPL, paths.file);
    }

    // Clean up logo - DISABLED FOR TESTING
    // var removed = GL.removeLogoByAbbr(t.abbr);
    // if (removed > 0) $.writeln("  ✓ Removed " + removed + " logo(s)");

    $.writeln("");
  }

  $.writeln("========================================");
  $.writeln("Batch complete: " + todo.length + " teams processed");
  $.writeln("========================================");

  app.endUndoGroup();
  if (app.endSuppressDialogs) { try { app.endSuppressDialogs(); } catch(e) {} }
  if (QUIT_APP) app.quit();
})();
